package furny.ga.rules;

import furny.entities.Tag;
import furny.ga.FurnEntry;
import furny.ga.FurnLayoutIndividual;
import ga.core.validation.GAContext;
import ga.core.validation.IValidationRule;

import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * This is a {@link IValidationRule} implementation that keeps the count of a
 * furniture in an interval.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class MinMaxCountRule implements IValidationRule<FurnLayoutIndividual> {

  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(MinMaxCountRule.class
      .getName());

  private final int minCount;
  private final int maxCount;
  private final Tag tag;

  /**
   * Creates the rule for a given tag name with given minimum and maximum
   * distances.
   * 
   * @param tagName
   *          The name of the tag.
   * @param minCount
   *          The minium count.
   * @param maxCount
   *          The maximum count.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public MinMaxCountRule(final String tagName, final int minCount,
      final int maxCount) {
    super();
    this.tag = Tag.valueOf(tagName);
    this.minCount = minCount;
    this.maxCount = maxCount;

    if (LOGGER.isLoggable(Level.FINE)) {
      LOGGER.fine(minCount + " <= n <= " + maxCount);
    }
  }

  @Override
  public float matchingDegree(final FurnLayoutIndividual individual,
      final GAContext context) {

    int i = 0;

    for (final FurnEntry entry : individual.getFurnitures()) {
      if (entry.getFurniture().getMetaData().getTags().contains(tag)) {
        i++;

        if (i > maxCount) {
          return 0;
        }
      }
    }

    if (i < minCount) {
      return 0;
    }

    return 1;
  }

  @Override
  public IValidationRule.Type getType() {
    return Type.PROPOSED;
  }
}
